/*
Spoof MIDI Channel 17 redirects notes to channel as function of octave

You can split you keyboard with one patch per octave

You can tune each note of the range of octaves independently.

* midiTemperament21.ino : tidy up and add Bank Select to Terminal Menu
* midiTemperament20.ino : working on patch settings & fixing button handling
* midiTemperament19.ino has a few improvements
* midiTemperament18.ino has working menu system that allows presetting a selection of temperaments
* Working version renamed midiTemperament16.ino for distribution
* Temper16.ino now accepts terminal commands
     - so don't need button, knob or LCDdisplay
	 - waiting for code to preload temperaments into designated channels

* Copyright 2012 Christopher R Lee <chrblee@gmail.com>
*
* Developed from:
* midiTranslator.ino,
* Copyright 2011 Warren Gill <mymaestro@gmail.com>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
* MA 02110-1301, USA.
*
*- - - - - - - - - - - 
* Using the MIDI library 3.2 found here
* http://www.arduino.cc/playground/Main/MIDILibrary with settings:

* #define COMPILE_MIDI_THRU       0           // Set this setting to 1 to use the MIDI Soft Thru feature
* #define USE_SERIAL_PORT         Serial1      // Serial(0) needed too; use Mega and snip two Serial pins of MIDI shield/breakout
*                                              // and leave shield/breakout switch in run position
* #define USE_RUNNING_STATUS		0			// Running status enables short messages when sending multiple values
*                                           // of the same type and channel.
*                                            // Set to 0 [2012-09-24]if you have troubles with controlling you hardware.
* #define USE_CALLBACKS           1           // Set this to 1 if you want to use callback handlers
* #define USE_1BYTE_PARSING       0           // [set to 0 2012-11-25] Each call to MIDI.read will only parse one byte (might be faster).
*
*
*
* This program takes MIDI signals arriving on any channel and forwards them
* to the channel assigned as the output channel
*
* When the channel is set to (non-existent) 17, notes are sent to channels
* as a function of their octave (multi-split).
* Using channel 17, a Roland JV 1010 can be tuned +-64 cents with each
* octave tuned differently. So you can go round the circle of fifths, play
* with ancient temperaments, and learn to tune a keyboard.
*
* It should be possible to adapt to different Roland instruments with similar
* memory map & sysex structures.
*
*/
#include <MIDI.h>
// #include <LiquidCrystal440.h>
#include <LiquidCrystal.h>
// #include <LcdBarGraph.h> used by midiTranslator
#include <limits.h>
#include <Bounce.h>

/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
Stuff the user may want to change
*/

//Conditional compilation
//The more directives you uncomment, the more verbose will be the output (Serial; terminal)
//#define DEBUG // The more you don't comment out the more verbose. May cause MIDI latency.
//#define READOUT
//#define NOTECOUNT
#define HARMONY  // Indicates harmony between 2 notes in "channel 17"
//#define INUSE (if commented out stops some of midiTranslator being compiled)
#define TUNING  // Compiles the parts that do tuning

const long baudRate = 115200;  // For Serial, used with the terminal (=terminal0).
                               // Go fast to avoid latency.   MIDI.h looks after Serial1


// range of octaves to be tuned when non-existent channel = 17 is selected
// 1 channel per octave
// This is also range available for multi-split (each octave can have a different patch/sound)
const byte lowestOctave = 1; //3 below middle C
const byte numberofOctaves = 6;
const byte startChannel = 11; //one octave per channel. Top chan = 16.

// notes to be used for tuning
// set these all to 1 to disable tuning function
const byte noteTuneOff = 101;     //F7  : stop adjusting but TuneSelect still active
const byte noteReset = 102;       // F#7 : reset note to default tuning value
const byte noteTuneDown = 103;   //G7 : down 1 cent each time note is played
const byte noteTuneSelect = 104; //G#7 : select a note to be tuned
const byte noteTuneUp = 105;     //A7 : up 1 cent each time note is played
const byte noteTuneCancel = 106; //A#7 : leave tuning mode, turn all notes off

//frequency setting of synthesiser (used to calculate beats)
//with Arduino double is same precision as float
const double freqA = 440;
// scale tune value is 64 for zero cents wrt E.T.; may need to change if transposing to 415Hz
const byte initTuneVal = 64;
//cents range for randomised notes
const byte randRange = 24;
//velocity when sounding control notes
const byte twiceVelocity = 80;
//temperament in use or to be set
int currentTemp = 0;
//note of preset temperaments to be fixed at zero cents.
//Scala default is C = 0; default here is A = 9, so your synth stays in
//tune with other instruments.
int zeroCentsNote = 9;  //int for use as array index

// Roland JVs have a scale tune per channel in performance mode only
// When in performance mode, Program Change messages (patch changes) must
// be preceeded by Bank Select (= Control Change to controllers 0 (MSB)
// & 32dec (LSB). This is done in void MidiProgramChange().
// You may need to chnage or remove these with other synths.
// The default below is for User patch group MSB 80dec, LSB 0.

byte bankSelMSB = 80;
byte bankSelLSB = 0;

// Tuning is in cents, declared as integers. This is accurate enough for 1 interval.
// Fortunately a 5th is 701.96 cents (nearly 702), so the Pythagorean comma
// for the circle of 12ths is nearly right (24, should be 23.46).

// Cents with respect to equal temperament for perfect intervals (integer ratios)
// Values are used to calculate (automatically!) the tuning accuracy whenever 2 notes are on.
// You can define only one interval for a given number of semitones.

// For Roland scale tune add 64 decimal (40h) and convert to byte (tuneHarmony below)
// Currently, tuneHarmony is not used in the program & is commented out.

const int centsHarmony[13] = {
0,   // 1:1 unison
12,  //16:15  minor second
4,   //9:8  major second
16,  //6:5  minor third
-14, //5:4  major third
-2,  //4:3  perf. fourth
-10, //45:32  augm. fourth (dim. fifth 64:45 = +10)
2,   //3:2  perf. fifth
14,  //8:5  minor sixth
-16, //5:3  major sixth
18,  //9:5  minor seventh
-12,  //15:8  major seventh
0 };  //2:1  perfect octave
/*
const byte tuneHarmony[13] = {
64,   // 1:1 unison
76,  //16:15  minor second
68,   //9:8  major second
80,  //6:5  minor third
50, //5:4  major third
62,  //4:3  perf. fourth
54, //45:32  augm. fourth (dim. fifth 64:45 = +10)
66,   //3:2  perf. fifth
78,  //8:5  minor sixth
48, //5:3  major sixth
82,  //9:5  minor seventh
52,  //15:8  major seventh
64 };  //2:1  perfect octave    */

// Change the text between "" according to your language and convention:

char* intervalNames[] = {
" 1:1 unison",
" 16:15  minor second ",
" 9:8  major second ",
" 6:5  minor third ",
" 5:4  major third ",
" 4:3  perf. fourth ",
" 45:32  augm. fourth ", // (dim. fifth 64:45 = +10)
" 3:2  perfect fifth ",
" 8:5  minor sixth ",
" 5:3  major sixth ",
" 9:5  minor seventh ",
" 15:8  major seventh ",
" 2:1  perfect octave " };
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
/*
Commonly used tunings that can be assigned to the current channel. They are copied from Scala where C is the reference note with zero cents.

If tuning to A, subtract algebraically the cents given here for A from all values. This is set as default by setting the variable: int zeroCentsNote = 9. Note that Scala can make the adjustment for calculating frequencies, but that is not the same thing.

Roland scale tune: add cents values to 64 decimal.*/

const int numberOfTemperaments = 18;  //can't use sizeof with pointer to temperamentNames

enum temperaments {  //not in use yet
eqTemp,
qMean,
thirdMean,
fifthMean,
sixthMean,
dAlambert,
vallotti,
werkIII,
werkIIIeb,
kirnIII,
rameau,
pythag12,
handel,
lehmanBach,
qMeanAflat,
spare1,
spare2,
spare3
} ;

char* temperamentNames[numberOfTemperaments] =
{
" Equal temperament",  //#0
" Quarter-comma meantone G# (wolf G#-Eb)",
" Third-comma meantone (Salinas)",
" Fifth-comma meantone (Holden)",
" Sixth-comma meantone (Salinas)",
" d'Alembert",
" Vallotti",
" Werckmeister III",
" Werkmeister III equal beating version; 5/4 beats twice 3/2",
" Kirnberger III",
" Rameau",
" Pythagorean 12-tone",
" Handel",
" Lehman's Bach",
" Quarter-comma meantone Ab (wolf C#-Ab), to be checked",
" Spare 1 (ET)",
" Spare 2 (ET)",
" Spare 3 (ET)"  //#17
};


//Scale tune values copied from Scala except lehmanBach & qMean (wolf C#-Ab)
int temperCents [numberOfTemperaments] [12] =
{
//eqTemp
{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,},
//qMean   wolf G#-Eb
{ 0, -24, -7, 10, -14, 3, -21, -3, -27, -10, 7, -17, },
//thirdMean
{ 0, -36, -10, 16, -21, 5, -31, -5, -42, -16, 10, -26, },
//fifthMean
{ 0, 12, -5, 7, -9, 2, -14, -2, 9, -7, 5, -12, },
//sixthMean
{ 0, -11, -3, 5, -7, 2, -10, -2, -13, -5, 3, -8, },
//dAlambert
{ 0, -22, -7, -18, -14, -6, -19, -3, -25, -10, -12, -16, },
//vallotti
{ 0, -6, -4, -2, -8, 2, -8, -2, -4, -6, 0, -10, },
//werkIII
{ 0, -10, -8, -6, -10, -2, -12, -4, -8, -12, -4, -8, },
//werkIIIeb
{ 0, -10, -7, -6, -8, -2, -12, -3, -8, -10, -4, -6, },
//kirnIII
{ 0, -8, -7, -6, -14, -2, -10, -3, -6, -10, -4, -12, },
//rameau
{ 0, -13, -7, -2, -14, 3, -15, -3, -11, -10, 7, -17, },
//pythag12
{ 0, 14, 4, -6, 8, -2, 12, 2, 16, 6, -4, 10, },
//handel
{ 0, -7, -5, -3, -4, -1, -7, -3, -5, -4, -2, -5, },
//lehmanBach
{ 0, -2, -4, -2, -8, 2, -4, -2, -2, -6, -2, -6, },
//qMean (wolf C#-Ab)
{ 0, -24, -7, 10, -14, 3, -21, -3, 14, -10, 7, -17, },
//spare1
{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,},
//spare2
{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,},
//spare3
{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}
} ;

//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  
/* Declarations in original midiTranslator program. Some are still used.  Nowadays we would declare constants instead of using #define, because constants are typed.  */
  
#define STATUS_LED 13 // LED pin on Arduino board
#define BACKLIGHT_CONTROL 12 // LCD backlight wired through ULN2803AN pin1
#define POWERSAVE 500000 // How many milliseconds to keep the backlight on
#define LCD_COLUMNS 16 // Number of columns on the LCD (originally 20)
#define LCD_ROWS 4 // Number of rows on the LCD
#define MIDI_PANIC_TIME 3000 // How long a button press to invoke the MIDI panic
#define RESET_BUTTON_TIME 5000 // How long the button has been pressed to initiate a reset
#define RX 19   //Serial1 pins on Mega used by MIDI.h
#define TX 18

#define encoder0PinA 2 // The rotary encoder is attached to pins 2 & 4 (was 2 and 4) int 0
#define encoder0PinB 4 // and ground of course

#define buttonPin 3 // Separate pushbutton (not on encoder) int 1; 

//Make Bounce library objet to debounce & read pushbutton
Bounce buttPush = Bounce( buttonPin, 100 );

const int startDelay = 2000;  // Initial message shown on LCD

// variables used by midiTranslator
int Pos, oldPos;
volatile unsigned int encoder0Pos = 0; // variables changed within interrupts are volatile
//volatile boolean buttonPressed;
boolean buttonPressed;
boolean buttonLast;
//long buttonDown;
long buttonPressTime;
long powerSave;

// must keep careful account of notes turned on/off on appropriate channels.
int notesOn = 0; // How many notes are on


// Draw an eighth note on the LCD
byte quaver[8] = { 0x02, 0x02, 0x03, 0x02, 0x02, 0x0E, 0x1E, 0x0C };

char* theNoteNames[] = { "C ", "C#", "D ", "Eb", "E ", "F ", "F#", "G ", "Ab", "A ", "Bb", "B "};

// The three kinds of parameters you can change
// Temper only uses Chan and Patch; 
// Asterisk : pointer 
char* midiParameterName[] = { "Chan: ",
                              "Patch: ",
                              "Volume: " };


// MIDI channel & messages
// Channel 17 is for multiple channels
char* midiChannelMessages[] = {
  "Starting; wait",
  "01",
  "02",
  "03",
  "04",
  "05",
  "06",
  "07",
  "08",
  "09",
  "10",
  "11",
  "12",
  "13",
  "14",
  "15",
  "16",
  "17 Tuning" };



int midiParameterIndex = 0; // Which parameter is changing
byte midiChannel = 1;       // User selected channel (1-16), 17 for multi
                            // MIDI library uses 1-16; converted to 0-15 for sysex
byte channelSent = 1;       // Channel sent will not be midiChannel if midiChannel = 17

//in MIDItranslator (patch number for each channel):
byte midiProgram[18];


// initialize the library with the numbers of the interface pins
LiquidCrystal lcd(7, 6, 8, 9, 10, 11); //Changed vs midiTranslator 


//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

//Variables for user input (commands) from Terminal (see Serial Event Arduino example)

String inputString = "";         // a string to hold incoming data
boolean stringComplete = false;  // whether the string is complete


// detecting notes to be used for tuning
boolean tuneDown = false;
boolean tuneSelect = false;
boolean tuneUp = false;
boolean tuneReset = false;
boolean tuneCancel = true;

// flag to say that next note played after noteTuneSelect will be the noteBeingTuned
boolean nextNote = false;

// noteBeingTuned is initialised below keyboard range
byte noteBeingTuned = 1;

// variables to calculate harmony whenever exactly 2 notes are on.
byte pitchHa[2] = {0, 0};
int h = 0;
boolean harmonyH = false;
boolean sysxArrived = false;
byte harmonyVal = 64;   //Roland scale tune value for zero cents

//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

/*
Stuff specific to Roland JV 1010
It should be easy to change this for other JV's & XP's if you have the sysex syntax and the memory map.

Variable for Roland parameter scale tune (64 is eq temp). JV has 1 value per note of scale, and one set of values per channel. It's the data byte of sysxSet message, and is updated by a sysex request and when tuning a note. It is reset by noteReset and the panic button command */
  byte tuneVal = 64;

//Set up scale tune SysEx according to make & model
//Identifiers can be modified for other Roland
const byte maKer = 0x41;
const byte devId = 0x10;  // can be changed if >1 identical synths
const byte modId = 0x6A;

/*For address values and their data to take effect the synthesiser JV1010 System parameter "scale tune switch" must be set to ON [1] at address
00 00 00 07. The sysex messages to turn the switch ON or OFF are below. For now the messages are not sent as the Roland has sysex ON already. */

//replace the constants by literals if necessary:
//const byte scaleTuneSwOn[] = { 0xF0, maKer, devID, modID, 0x12, 0x00, 0x00, 0x00, 0x07, 0x01, 0x78, 0xF7 };
//const byte scaleTuneSwOff[] = { 0xF0, maKer, devID, modID, 0x12, 0x00, 0x00, 0x00, 0x07, 0x00, 0x79, 0xF7 };

//check if this variable is actually needed:
byte sysexChannel = 11;  // current channel for setting up sysex messages

// Constants below not used everywhere in program (parsing Roland sysx can be tricky)
// The lengths include F0 and F7 (see MIDI library)

const int sysxLengthReq = 15;  // requesting 1 byte, but sending all 4 size bytes
const int sysxLengthSet = 12;  // here, Roland returns only 1 tuning data byte, so length is shorter than sysxLengthReq
const int sysxLengthPreset = 23; // sends 12 scale tuning bytes, one sysex sent per octave in tuning range. Needs > 20ms delay.

byte sysxReq[sysxLengthReq] = {
  0xF0,
  maKer,  //Maker
  devId,   //Device id
  modId,   //Model id
  0x11,  //Data request 1
  0,        //Addr MSB
  0,
  0,        // index = 7
  0x1A,     //Addr LSB index = 8
  0,
  0,
  0,
  0x1,      //Size (number of bytes requested, always 1 for this Req
  0,     // Checksum to be calculated; index = 13d
  0xF7 };
  
byte sysxSet[sysxLengthSet] = {
  0xF0,
  maKer,   //Maker
  devId,   //Device id
  modId,   //Model id
  0x12,    //Data transmit 1
  0,       //Addr MSB
  0,
  0,       //index = 7
  0,       //Addr LSB; index = 8
  0,       //Data (there will be only one byte, assigned to tuneVal)
  0,       // Checksum to be calculated; index = 10d
  0xF7 };
  
  byte sysxPreset[sysxLengthPreset] = {
  0xF0,
  maKer,   //Maker
  devId,   //Device id
  modId,   //Model id
  0x12,    //Data transmit 1
  0,       //1 -> 5 Addr MSB
  0,       //2
  0,       //3  -> 7
  0,       //4  -> 8 addr LSB
  
  0,       //1  -> 9  Data (12 bytes, the scale tune values to be sent)
  0,       //2
  0,      0,  0,  0,  0,  0,  0,  0,  0,
  0,       //12 -> 20
  
  0,       // Checksum to be calculated; index = 21d
  0xF7 };
  
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 
void setup() {
  Serial.begin(baudRate);
  
  buttonPressed = false;
  buttonLast = false;
  // Pins for status LED and the LCD backlight
  pinMode(STATUS_LED, OUTPUT);
  pinMode(BACKLIGHT_CONTROL, OUTPUT);
  // Initiate pins for rotary encoder and button
  pinMode(encoder0PinA, INPUT);
  digitalWrite(encoder0PinA, HIGH); //Activates pullup. See buttonPin for version 1.0.1 alternative cmd
  pinMode(encoder0PinB, INPUT);
  digitalWrite(encoder0PinB, HIGH);
  //pinMode(buttonPin, INPUT_PULLUP); //now handled by Bounce.h.
  pinMode(TX, OUTPUT);  
  pinMode(RX, INPUT);
  digitalWrite(RX, HIGH);

  // Set up the rotary encoder
  attachInterrupt(0, doEncoder, CHANGE); // encoder pin on interrupt 0 (pin 2)

  // Initiate MIDI communications, listen to all channels, turn off thru mode
  MIDI.begin(MIDI_CHANNEL_OMNI);
  /* By default, the MIDI library sends everything THRU. We do NOT want that! Either call MIDI.turnThruOff(); in your setup or #define COMPILE_MIDI_THRU 0 in the library's MIDI.h (latter done) */

  // Connect MIDI status changes involving a channel to handlers
  // Uncomment the ones you bring into use. Handlers are also commented out at end of program.
  MIDI.setHandleNoteOn(HandleNoteOn);
  MIDI.setHandleNoteOff(HandleNoteOff);
  //MIDI.setHandleAfterTouchPoly(HandleAfterTouchPoly);
  MIDI.setHandleControlChange(HandleControlChange);
  MIDI.setHandleProgramChange(HandleProgramChange);
  //MIDI.setHandleAfterTouchChannel(HandleAfterTouchChannel);
  //MIDI.setHandlePitchBend(HandlePitchBend);
  MIDI.setHandleSystemExclusive(HandleSystemExclusive);

  turnBacklightOn();
  // set up the LCD's number of columns and rows:
  lcd.begin(LCD_COLUMNS, LCD_ROWS);
  lcd.createChar(5, quaver); // it's a picture of a note
  
  // Print a message to the LCD.
  lcd.home();
  EnterSetupMenu();
  
  // reserve 16 bytes for the users' Terminal inputString:
  inputString.reserve(16); 
  
  delay(500);
  Serial.println("Starting");
  for (int im = 0; im < 10; im++) {Serial.println(" ");}
  Serial.println("Send any key for menu"); Serial.println(" ");
  Serial.println("All other commands are numeric");
  
  /* Example of reading out temperament settings
  for (int i = 0; i < 12; i++) {
	Serial.print(temperCents [qMean] [i]); Serial.print(" "); }
  Serial.print("qMean "); Serial.println(qMean);	*/
}


void loop() {
  MIDI.read();
  
  // Rotary encoder detection
  // Turn off the interrupt while scanning it
  uint8_t oldSREG = SREG;
  cli();
  Pos = encoder0Pos;
  SREG = oldSREG;
  // The rotary encoder has turned
  if (Pos != oldPos) HandleEncoderTurn();
  
  // The button has changed since last loop (now not interrupt driven)
  if (buttPush.update())  { 
#ifdef DEBUG  
  int readVal = buttPush.read();
  Serial.print("Button state : "); Serial.println(readVal);
#endif  
  HandleButtonPress(); 
  }
  
  // Turn the back light off after timeout
  if (millis() > powerSave) turnBacklightOff();
  
   // print the user's string when a newline arrives from Terminal:
  if (stringComplete) {
    //Serial.println(inputString);
	terminalAction();
    // clear the string:
    inputString = "";
    stringComplete = false;
  }
  
  // Action if exactly 2 notes are on at the end of HandleNoteOn()
	if(harmonyH == true) {
    HarmonyCalc();
	harmonyH = false;
   }
   
   //See if a MIDI command has arrived; a command will be sent to the appropriate handler
   MIDI.read();
}

void terminalAction() {
  
  //Responds to input via terminal. Any key to start, then input must be numeric. If non numeric, 9999 will be returned.
  
  // Static variables will be initialised only the first time terminalAction is called, and held in memory thereafter:
  static int terminalLevel = 0;   // For hierarchy of terminal commands
  static int inputNumber = 0; //Stores last number input from Terminal
  static int inputMem = 0; // Stores an earlier number when needed
  
  byte presetType;
  
  inputNumber = aStringToInt(inputString);
  if (terminalLevel != 0 && inputNumber == 9999) {
	Serial.println("Expecting a number; returning to command level 0");
	terminalLevel = 0;
	return; }
  
  if (inputNumber < 9999) {
  Serial.println();
  Serial.print("Last number sent = "); Serial.print(inputNumber);
  Serial.print("  Command level = "); Serial.println(terminalLevel); }

    switch (terminalLevel) {
		case 0:
		Serial.println("Make your choice: "); Serial.println();
		Serial.print("1: set channel (MIDI = 1-16, Tuning = 17 ; ");
		Serial.print("Currently ");Serial.println(midiChannel);
		Serial.println(""); Serial.println("Settings for current channel, or channel range if pseudo-channel 17 (tuning) :");
		Serial.println("2: set equal temperament");
		Serial.println("3: randomise tuning");
		Serial.println("4: choose a preset temperament");
		Serial.println("5: choose a patch (1-128)");
		Serial.println("6: select a tonebank");
		terminalLevel = 1;
		break;  // from TerminalLevel 0
	
		case 1:	 //terminalLevel is 1
				
			switch (inputNumber) {
			
			case 1:  //setting MIDI channel
			inputMem = 1;
			Serial.println("Choose your channel (MIDI = 1-16, Tuning = 17)");
			terminalLevel = 2;
			break;
			
			case 2:  //setting equal temperament
			presetType = 1;
			currentTemp = 0;
			PresetTuning(presetType);
			terminalLevel = 0;
			break;
			
			case 3:  //randomising tuning; we could allow choice of range
			presetType = 2;
			currentTemp = 0;
			PresetTuning(presetType);
			terminalLevel = 0;
			break;
			
			case 4:  //choose a preset temperament
			inputMem = 4;
			Serial.println(); Serial.println("Choose your temperament "); Serial.println();
			for (int t = 0; t < numberOfTemperaments; t++) {  // (< numberOfTemperaments as count is from zero)
				Serial.print(t); Serial.print(" : "); Serial.println(temperamentNames[t]); }
			Serial.println();
			terminalLevel = 2;
			break;
			
			case 5:  //choose a patch
			inputMem = 5;
			Serial.println(); Serial.println("Choose your patch (1-128)"); Serial.println();
			terminalLevel = 2;
			break;
			
			case 6:  //bank select (2 controllers)
			inputMem = 6;
			Serial.println(); Serial.println("Send MSB (preset is 80 decimal)"); Serial.println();
			terminalLevel = 2;
			break;
			
			default:
			terminalLevel = 0;
			break;
			}
		break;  //from terminalLevel 1
		
		case 2: //terminalLevel
			switch (inputMem) {
			
			 case 1:  //Changing channel
			 ChannelChange(inputNumber);
			 terminalLevel = 0;
			 break;
			 
			 case 4:  //preset temperament
			 presetType = 3;
			 currentTemp = inputNumber;
			 PresetTuning(presetType);
			 terminalLevel = 0;
			 break;
			 
			 case 5:  //patch change
			 midiProgram[midiChannel] = byte(lowByte(inputNumber));
			 MidiProgramChange(midiChannel);
			 terminalLevel = 0;
			 break;
			 
			 case 6:  //bank select (2 controllers)
			 bankSelMSB = byte(lowByte(inputNumber));
			 Serial.print("Tonebank MSB for next patch change will be: "); Serial.println(bankSelMSB);
			 Serial.println(); Serial.println("Send LSB (preset is 0)");
			 Serial.println();
			 
			 inputMem = 6;  //confirmation, probably not needed
			 terminalLevel = 3;
			 break;
			 
			 default:
			 terminalLevel = 0;
			 break;
			 }
		break;  //from Terminal Level 3
		
		case 3:   //terminalLevel
			switch (inputMem) {
			
			case 6:  //bank select (2 controllers)
			bankSelLSB = byte(lowByte(inputNumber));
			Serial.print("Tonebank LSB for next patch change will be: "); Serial.println(bankSelMSB);
			terminalLevel = 0;
			break;
			 
			 default:
			 terminalLevel = 0;
			 break;
			}
		
		break; }
	      
}

int aStringToInt(String inString) {

//only reliable way of extracting an int from an Arduino String ??
  char buffer[20];
  char* endptr;
  long value;

	inString.trim();
	inString.toCharArray(buffer, sizeof(buffer));
	//char Cstring[] = buffer
    buffer[sizeof(buffer)-1] = 0;  
    value = strtol( buffer, &endptr, 0 );
	if ( endptr != &buffer[0] && value > INT_MIN && value < INT_MAX)
		{
			//Serial.println( value );
			return int(value);
		}
  else
  {	return 9999;   }	
 }

 
void doBlink(int times) {
  for (int i = 0; i < times; i++ ) {
    digitalWrite(STATUS_LED, HIGH);
    delay(150);
    digitalWrite(STATUS_LED, LOW);
    delay(150);
  }
}

void doButtonPress()
{
  buttonPressed = !buttonPressed;
}

void HandleButtonPress() {

static long buttonDown;

  //if (buttonPressed != buttonLast) { // The button changed since last
  //there's a logic inversion somewhere (PULLUP?)
    if (buttPush.read() == HIGH) { // The button is down
      buttonDown = millis(); // Start button down timing
      lcd.setCursor(13, 2); // was 15,2 in midiTranslator; using 16 cols
      lcd.write(5);
      turnBacklightOn();
	  buttonPressed = true;
    } else { // The button is up
      buttonPressed = false;
	  buttonPressTime = millis() - buttonDown; // How long the button was down
      lcd.setCursor(13, 2);
      lcd.print(" ");
      if ( Pos = oldPos ) { // The encoder did not turn
        // If the button was pressed a long time but didn't turn, perform
        // the MIDI panic button or a reset, depending on how long
        if (buttonPressTime > RESET_BUTTON_TIME) {
          EnterSetupMenu();
        }
        if ((buttonPressTime > MIDI_PANIC_TIME) && (buttonPressTime < RESET_BUTTON_TIME)) {
          midiPanicButton();
        }
      }
    }
  
}
void HandleEncoderTurn() {
  
  turnBacklightOn();
  if(!buttonPressed) { // encoder is turning but not pressed in
    switch (midiParameterIndex) {
      case 0: // Channel
        if(Pos > oldPos) { // turning clockwise
          midiChannel++;
          if (midiChannel > 17) midiChannel = 1;  //Chan 17 goes into tuning mode
        } else { // turning counterclockwise
            midiChannel--;
            if (midiChannel < 1) midiChannel = 17;
        }
        
		displayChannelMessage(midiChannel);
		if (midiChannel < 17) {
		displayProgramMessage(midiProgram[midiChannel]);
        #ifdef INUSE		  
          vvcdisplayVolumeMessage(midiVolume[midiChannel]);
		#endif
		}
        break;
		
	    case 1: // Program
        if (Pos > oldPos) { // turning clockwise
          midiProgram[midiChannel]++;
          if (midiProgram[midiChannel] > 128) midiProgram[midiChannel] = 1;
        } else { // turning counterclockwise
          midiProgram[midiChannel]--;
          if (midiProgram[midiChannel] < 1) midiProgram[midiChannel] = 128;
        }
		
		MidiProgramChange(midiChannel);
		
        break;  
		
      }
	  
      delay(20);
	  
     } 

	else { // button being held down ; Temper: don't change midiParameterIndex
      lcd.setCursor(0, midiParameterIndex);
      lcd.print(" ");
      if(Pos > oldPos) { // turning clockwise
        midiParameterIndex++;
        if (midiParameterIndex > 1) midiParameterIndex = 0;
      } else { // turning counterclockwise
        midiParameterIndex--;
        if (midiParameterIndex < 0) midiParameterIndex = 1;
      }
      lcd.setCursor(0, midiParameterIndex);
      lcd.write(5);
      delay(20);
    } 
	
  oldPos = Pos;
}

void MidiProgramChange(byte midiChannel)  {
int intChannel = int(midiChannel);

if (midiChannel < 17) {
		
			MIDI.sendControlChange(0, bankSelMSB, midiChannel);
			delay(50);
			MIDI.sendControlChange(32, bankSelLSB, midiChannel);
			delay(50);
		MIDI.sendProgramChange(midiProgram[intChannel] - 1, midiChannel);
		delay(50);
        displayProgramMessage(midiProgram[intChannel]); }
		
		else {
			
			for (int m = 0 ; m < numberofOctaves; m++)  {
				int chanM = int(startChannel) + m;
				
				if (chanM > 16) {chanM = 16;}
				byte chanB = byte(chanM);
				midiProgram[chanM] = midiProgram[17];
				Serial.print("Channel 17 ; chanB "); Serial.println(chanB);
				MIDI.sendControlChange(0, bankSelMSB, chanB);
				delay(50);
				MIDI.sendControlChange(32, bankSelLSB, chanB);
				delay(50);
				MIDI.sendProgramChange(midiProgram[chanM] - 1, chanB);
				delay(50);
				displayProgramMessage(midiProgram[intChannel]);
			}
		}  
Serial.println(" "); Serial.print("Patch : ");
Serial.println(midiProgram[intChannel]); Serial.println();

}





/*
  SerialEvent occurs whenever a new data item comes in the
 hardware serial RX (user input from Terminal).  This routine is run between each
 time loop() runs, so using delay inside loop can delay
 response.  Multiple bytes of data may be available.
 */
void serialEvent() {
  while (Serial.available()) {
    // get the new byte:
    char inChar = (char)Serial.read();
    // add it to the inputString:
    inputString += inChar;
    // if the incoming character is a newline, set a flag
    // so the main loop can do something about it:
    if (inChar == '\n') {
      stringComplete = true;
    }
  }
}

/*
* Sometimes the synth gets a noteOn but loses the noteOff, and the note keeps playing forever
* These two channel control messages turn everything off, commonly called the "MIDI Panic" button
*
* CC 120 is All Sound Off
* CC 123 is All Notes Off

* Also called when resetting tuning via MIDI keyboard
*/
void midiPanicButton() {
  lcd.setCursor(0, 3);
  lcd.print("ALL NOTES OFF");
  for (int channel=1; channel <= 16; channel++) {
    MIDI.sendControlChange(120, 0, channel);
    MIDI.sendControlChange(123, 0, channel);
  }
  notesOn = 0;
}

void turnBacklightOn() {
  powerSave = millis() + POWERSAVE;
  digitalWrite(BACKLIGHT_CONTROL, HIGH);
}

void turnBacklightOff() {
  digitalWrite(BACKLIGHT_CONTROL, LOW);
}

void EnterSetupMenu() {
  // Clear the screen and print the welcome message
  lcd.clear();
  // Print a message to the LCD.
  lcd.setCursor(0, 0);
  lcd.print(midiChannelMessages[0]);
  lcd.setCursor(0, 1);
  lcd.print("Turn: ch set");
  lcd.setCursor(0, 2);
  lcd.print("Press+turn:");
  lcd.setCursor(0, 3);
  lcd.print(" Ch param");
  // Rest the indices, channel, volume, and program settings
  midiParameterIndex = 0;
  midiChannel = 1;
   
  // Wait xx seconds to read the screen
  delay(startDelay);
  lcd.clear();
  lcd.write(5);
  lcd.setCursor(1, 0);
  displayChannelMessage(midiChannel);
  displayProgramMessage(midiProgram[midiChannel]); //(midiProgram[midiChannel]);
  //displayVolumeMessage(midiVolume[midiChannel]);
  buttonPressed = false;
}



  void displayChannelMessage(int messageNum) {
  lcd.setCursor(0, 0);
  for (int i=0; i < LCD_COLUMNS; i++) {lcd.print(" ");} //clear row zero
  lcd.setCursor(0, 0);
  lcd.write(5); //[need to check if this is OK every time function is run]
  lcd.print(midiParameterName[0]);
  lcd.print(midiChannelMessages[messageNum]);
}


void displayProgramMessage(byte mProgram) {
//int intChannel= int(midiChannel);
  lcd.setCursor(1, 1);
  lcd.print(midiParameterName[1]);
  lcd.print(mProgram);
  //lcd.print(midiProgram[intChannel]);
  lcd.print(" ");
}


/* See this expanded function to get a better understanding of the
* meanings of the four possible (pinA, pinB) value pairs:
*/
void doEncoder(){
  delay(4);
  if (digitalRead(encoder0PinA) == HIGH) { // found a low-to-high on channel A
    if (digitalRead(encoder0PinB) == LOW) { // check channel B to see which way
      // encoder is turning
      encoder0Pos = encoder0Pos - 1; // CCW
    } else {
      encoder0Pos = encoder0Pos + 1; // CW
    }
  } else { // found a high-to-low on channel A
    if (digitalRead(encoder0PinB) == LOW) { // check channel B to see which way
      // encoder is turning
      encoder0Pos = encoder0Pos + 1; // CW
    } else {
      encoder0Pos = encoder0Pos - 1; // CCW
    }
  }
}

void HandleNoteOn(byte channel, byte pitch, byte velocity)
{
  #ifdef DEBUG
   Serial.print("Note count when entering HandleNoteOn  ");
   NotesOnCount();
  #endif
  
  channelSent = midiChannel;
  
  //If 'note on' with zero velocity arrives here in HandleNoteOn
  // ...send it to HandleNoteOff
  if (velocity == 0) {
 #ifdef DEBUG
   Serial.print("Note off velocity: ");  Serial.println(velocity);
 #endif
  HandleNoteOff(midiChannel, pitch, velocity); 
 
 #ifdef DEBUG
   Serial.print("Note count when velocity = zero  ");
   NotesOnCount();
 #endif
   return;  }
   
//channelSent will not be a midiChannel (selected by rotary) if 17 
  if (midiChannel < 17) {    //not in tuning mode
      #ifdef DEBUG
	  Serial.print("Notes on when midiChannel < 17 "); Serial.println(notesOn);
	  #endif
     SendNoteOnce(pitch, velocity, midiChannel);
     return;
  }
   
 // all the rest is for tuning mode ("channel 17")
 // note played just after noteTuneSelect is selected
  if (nextNote == true) {
  Serial.println("Is Next note");
  noteBeingTuned = pitch;
  nextNote = false;
  channelSent = MultiChan(pitch);
  SendNoteTwice(pitch, velocity, channelSent);
  // put data in sysex request command & calc. chksum
  SetSysxReqAdr(pitch);
#ifdef READOUT  
  Serial.print("sending sysex request, size ");
  Serial.println(sizeof(sysxReq));
  Serial.println(" ");
#endif  
  MIDI.sendSysEx(sizeof(sysxReq), sysxReq);
  delay(50);
  return;}

// what to do if pitch is one of the command notes
  switch(pitch)
  {
    case noteTuneOff:
	Serial.println("noteTuneOff");
	tuneDown = false;
	tuneUp = false;
	tuneReset = false;
	channelSent = MultiChan(pitch);
    SendNoteTwice(pitch, velocity, channelSent);
	break;
	
	case noteTuneDown:
	Serial.println("noteTuneDown");
    if (tuneSelect == true) {
	tuneDown = true;
    tuneUp = false;
	tuneReset = false;
    tuneCancel = false; 
	channelSent = MultiChan(pitch);
    SendNoteTwice(pitch, velocity, channelSent);}
    break;
	
    case noteTuneSelect:
	Serial.println("noteTuneSelect");
    tuneDown = false;
    tuneSelect = true;
	nextNote = true;
    tuneUp = false;
	tuneReset = false;
    tuneCancel = false;
	channelSent = MultiChan(pitch);
    SendNoteTwice(pitch, velocity, channelSent);
    break;
	
	case noteTuneUp:
	Serial.println("noteTuneUp");
    if (tuneSelect == true) {
	tuneDown = false;
    tuneUp = true;
    tuneCancel = false;
	tuneReset = false;
	channelSent = MultiChan(pitch);
    SendNoteTwice(pitch, velocity, channelSent); }
    break;
	
	case noteReset:
	Serial.println("noteReset");
	if (tuneSelect == true) {
	tuneReset = true;
	channelSent = MultiChan(pitch);
    SendNoteTwice(pitch, velocity, channelSent); }
    break;
	
    case noteTuneCancel:
	Serial.println("noteTuneCancel");
    tuneDown = false;
    tuneSelect = false;
    tuneUp = false;
    tuneCancel = true;
    channelSent = MultiChan(pitch);
    SendNoteTwice(pitch, velocity, channelSent);
	delay(300);
    midiPanicButton();  //notes off
    notesOn = notesOn + 1;  //cos panic sets to zero then note release decrements
    break;
    
    default:
	// pitch is note to be adjusted
        if (pitch == noteBeingTuned && nextNote == false) {
	         #ifdef DEBUG
			 Serial.println("noteBeingTuned");
			 #endif
	         if (tuneSelect == true) {
	            AdjustNote(pitch, velocity);}
			else {channelSent = MultiChan(pitch);
                  SendNoteOnce(pitch, velocity, channelSent);}
		}
        else {     //another note just to be played
          channelSent = MultiChan(pitch);
          SendNoteOnce(pitch, velocity, channelSent);}
#ifdef DEBUG
            Serial.print("pitch ");
            Serial.println(pitch);
#endif
        break;
}

#ifdef HARMONY
    pitchHa[h] = pitch;
	h++;
    if (h > 1) {h = 0;}
	
    if (notesOn == 2) {
    harmonyH = true;
    //Serial.print("Harmony in HandleNoteOn "); Serial.println(harmonyH);
	}
#endif
	//This is the end of HandleNoteOn, at last
	}
	
   void AdjustNote(byte pitch, byte velocity)
   {
   #ifdef DEBUG
   Serial.println("Entering AdjustNote");
   #endif
   //tuneVal is a byte with MSB = 0
   if (tuneUp) {
   tuneVal=tuneVal+1;
   if (tuneVal > 127) {
     tuneVal = 0; }
	#ifdef DEBUG 
   Serial.println("up");
    #endif
 }
   if (tuneDown) {
   tuneVal=tuneVal-1;
   if (tuneVal > 127) {
     tuneVal = 127; }
	 #ifdef DEBUG
     Serial.println("down");
     #endif
}
   if (tuneReset) {
   tuneVal=64;
   Serial.println("reset"); 
 }
   //put data in sysxSet and do chksum
   sysxSet[9] = tuneVal;
   channelSent = MultiChan(pitch);
   SetSysxSetAdr(channelSent, pitch);
              #ifdef DEBUG
			Serial.println("sysxSet");
			for (int i=0; i<sizeof(sysxSet);i++) {
			Serial.print(sysxSet[i], HEX); 
			Serial.print(" ");}
			Serial.println(" ");
			  #endif
   MIDI.sendSysEx(sizeof(sysxSet), sysxSet);
   delay(50);
   SendNoteOnce(pitch, velocity, channelSent);
   #ifdef TUNING
     Serial.print("Pitch: ");
     Serial.print(pitch); Serial.print(" ");
	 Serial.print(theNoteNames[noteIndex(pitch)]);
	 //Serial.print(" ");
	 Serial.print(noteOctave(pitch));
	 Serial.print(" Channel: "); Serial.print(channelSent); 
     Serial.print("  Cents vs ET: ");
     Serial.println(tuneCents(tuneVal)); Serial.println();
   #endif
   }
  
void SendNoteOnce(byte pitch, byte velocity, byte channelSent)
{
  MIDI.sendNoteOn(pitch, velocity, channelSent);
  int noteIndex = (pitch - 12) % 12; //could call the function
  int noteOctave = (pitch / 12) - 1;
  turnBacklightOn(); // Turn the light on when a MIDI key is pressed
    lcd.setCursor(0, 2);
    lcd.print("[");
    lcd.print(theNoteNames[noteIndex]);
    lcd.print(noteOctave);
    lcd.print("]");
  #ifdef DEBUG
    Serial.print(theNoteNames[noteIndex]); Serial.print(" ");
    Serial.println(noteOctave);
  #endif
  
  notesOn++;
  
  #ifdef NOTECOUNT
    Serial.print("Note count in SendNoteOnce  ");
    NotesOnCount();
  #endif
}

void SendNoteTwice(byte pitch, byte velocity, byte channelSent)
//twiceVelocity is constant set so the notes are played softly
{
#ifdef DEBUG  
  Serial.println("Twice!");
#endif
  SendNoteOnce(pitch, twiceVelocity, channelSent);
  delay(200);
  MIDI.sendNoteOff(pitch, twiceVelocity, channelSent);
  notesOn--;
  delay(200);
  SendNoteOnce(pitch, twiceVelocity, channelSent);
#ifdef NOTECOUNT 
 // N.B. latency doesn't matter here
  NotesOnCount();
#endif
} 

byte MultiChan(byte pitch) {
  // when in channel 17, finds the channel to send the note or sysex to
  
  // find the channel (octave) to which the pitch is assigned
  byte chanOctave = (pitch / 12) - 1; //middle C = C4; 60d 3Ch
  byte chan;
  if (chanOctave >= lowestOctave && chanOctave <= (lowestOctave + numberofOctaves))
  {  chan = startChannel - lowestOctave + chanOctave;  }
  //otherwise return lowest channel number defined by a constant
  else {chan=startChannel;}
  //was > 15 in earlier versions (channels are 1-16)
  if (chan > 16) {chan = startChannel;}
  return chan;
 }


void HandleNoteOff(byte channel, byte pitch, byte velocity) {
  //called by callback only if "genuine" note off starting with 8
  // note on with velocity zero is handled in HandleNoteOn which passes it here
  #ifdef DEBUG
    Serial.println("Entering HandleNoteOff");
  #endif
  if (midiChannel < 17)
  {
    MIDI.sendNoteOff(pitch, velocity, midiChannel);
  notesOn--;
  #ifdef DEBUG
    Serial.print("Note off, ch < 17  ");
  #endif
  #ifdef NOTECOUNT
    NotesOnCount();
  #endif
   }
  else 
   {channelSent = MultiChan(pitch);
   MIDI.sendNoteOff(pitch, velocity, channelSent);
   notesOn--;
   #ifdef DEBUG
     Serial.print("Note off, ch = 17  ");
   #endif
   #ifdef NOTECOUNT
     NotesOnCount();
   #endif
   }
}

void PresetTuning(byte presetType) {   
// Handles terminal input by Serial
// Presets tuning over declared range of octaves
// presetType = 1 : set equal temperament
// presetType = 2 : randomise tuning
// presetType = 3 : set the chosen temperament (temperamentNames, temperCents)

// Spoof channel 17 is for tuning  

byte chanPreset = 0; 
#ifdef DEBUG
Serial.println("PTuning "); //Serial.println(inputString);
#endif

if (presetType > 0 && presetType < 4) {  // change limits if necessary
    
	if (midiChannel != 17)  {
		chanPreset = midiChannel;
		SetSysxPresetAdr(chanPreset, presetType);
		MIDI.sendSysEx(sizeof(sysxPreset), sysxPreset);
		delay(50); }  // need a longish delay because of number of bytes  }
	else  {
		for (int i = 0 ; i < numberofOctaves; i++) {
		chanPreset = startChannel + i;
		if (chanPreset > 16) {chanPreset = 16;}
		SetSysxPresetAdr(chanPreset, presetType);
		MIDI.sendSysEx(sizeof(sysxPreset), sysxPreset);
		delay(50); }  // need a longish delay because of number of bytes
		}
	return;  }
}

void ChannelChange(int chanSet) {
if (chanSet > 0 && chanSet < 18)  {
	Serial.print("Setting channel by terminal command to ");
	Serial.println(chanSet); Serial.println();
	midiChannel = lowByte(chanSet);
	displayChannelMessage(midiChannel); }
else {Serial.println("Channel out of range 1 - 17"); }
}

void CheckSumReq() {
  // specific for this app: 4 mem bytes (5-8), 4 size bytes (9-12)
  unsigned int ckSum = 0;
    #ifdef DEBUG
	Serial.println("Before CheckSumReq calculation");
    for (int j=0; j < sysxLengthReq; j++) {
    Serial.print(sysxReq[j], HEX);
    Serial.print(" ");}
	Serial.println(" "); Serial.println(" "); 
	#endif
  
for (int i=5; i < 13; i++) {            
  ckSum = ckSum + (unsigned int)sysxReq[i];             
}
  ckSum &= 0x7F;  //bitwise AND retains only low byte
  if( ckSum != 0 )
  {ckSum = 0x80 - ckSum;}
   // MIDI library uses the Arduino type (byte)
  sysxReq[13] = (byte)ckSum;
    #ifdef DEBUG
	Serial.println("After CheckSumReq calculation");
    for (int j=0; j < sysxLengthReq; j++) {
    Serial.print(sysxReq[j], HEX);
    Serial.print(" ");}
	Serial.println(" ");
	#endif
}


void CheckSumSet() {
  // specific for this app: 4 mem bytes (5-8), 1 data (9)
  // JV sysxReq returns only 1 data byte in this case
  unsigned int ckSum = 0;
    
    #ifdef DEBUG
	Serial.println("Before CheckSumSet calculation");
	for (int j=0; j < sysxLengthSet; j++) {
    Serial.print(sysxSet[j], HEX);
    Serial.print(" ");}
	Serial.println(" "); Serial.println(" "); 
	#endif
  
for (int i=5; i < 10; i++) {            
  ckSum = ckSum + (unsigned int)sysxSet[i]; }
  ckSum &= 0x7F;
  if( ckSum != 0 )
  {ckSum = 0x80 - ckSum;}
   // MIDI library uses the Arduino type (byte)
  sysxSet[10] = (byte)ckSum;
  #ifdef DEBUG
    for (int j=0; j < sysxLengthSet; j++) {
    Serial.print(sysxSet[j], HEX);
    Serial.print(" ");}
	Serial.println(" "); Serial.println(" "); 
  #endif
}


void CheckSumPreset() {
  // specific for this app: 4 mem bytes (5-8), 12 data (9-20)
  // 
  unsigned int ckSum = 0;
    
    #ifdef DEBUG
	Serial.println("Before CheckSumPreset calculation");
	for (int j=0; j < sysxLengthPreset; j++) {
    Serial.print(sysxPreset[j], HEX);
    Serial.print(" ");}
	Serial.println(" "); Serial.println(" "); 
	#endif
  
// do checksum from 5 (counting from zero) to last but 2
  for (int i=5; i < sysxLengthPreset - 2; i++) {            
  ckSum = ckSum + (unsigned int)sysxSet[i]; }
  ckSum &= 0x7F;
  if( ckSum != 0 )
  {ckSum = 0x80 - ckSum;}
   // MIDI library uses the Arduino type (byte)
   // put checksum in last but one position:
  sysxPreset[sysxLengthPreset - 2] = (byte)ckSum;
  //#ifdef READOUT
    Serial.println("After CheckSumPreset calculation");
	for (int j=0; j < sysxLengthPreset; j++) {
    Serial.print(sysxPreset[j], HEX);
    Serial.print(" ");}
	Serial.println(" "); Serial.println(" "); 
  //#endif
}



void HandleSystemExclusive(byte *sxArray, byte sxSize) {
/* Incoming array looks like sysxSet (no leading data zeros)  & we want the byte of data,
which has index last-2 = sxSize-3. Could also just put in the 
number (9 here); use global tuneVal in case index is changed later */
#ifdef DEBUG
 Serial.print("harmonyH entering handlesysex  "); Serial.println(harmonyH);
#endif

if (harmonyH == true)
    {harmonyVal = sxArray[sxSize-3];
	
#ifdef READOUT	
	Serial.print("harmonyVal in HandleSyx  "); Serial.println(harmonyVal);
	Serial.println("");
#endif
	}
else
	{tuneVal= sxArray[sxSize-3];}
#ifdef READOUT
  Serial.print("tuneVal from incoming sysex ");
  for (int j=0; j < int(sxSize); j++) {
    Serial.print(sxArray[j], HEX); //note final F7 is zero or absent
    Serial.print(" ");}
	Serial.print("sxSize (dec) ");
	Serial.println(sxSize);
    Serial.println(" ");
  Serial.print("tuneVal ");
  Serial.print(tuneVal);
  Serial.print("  cents ");
  Serial.println(tuneCents(tuneVal));
  Serial.println(" ");
#endif  
if (harmonyH == false)
    {sysxSet[9] = tuneVal;}
sysxArrived = true;	
}

void SetSysxReqAdr(byte pitch) {
// set 3rd addr byte to match channel, offset 10h (Roland JV)
// subtract 1 (channels 1-16 --> 0-15)
byte channelData = MultiChan(pitch)-1;
sysxReq[7] = channelData + 0x10;
//set 4th addr byte to match note, offset 0
sysxReq[8] = (pitch - 12) % 12;
// calculate and store checksum
CheckSumReq();
}

void SetSysxSetAdr(byte channelSent, byte pitch) {
#ifdef DEBUG
  Serial.println("Entering SetSysxSetAdr");
  Serial. println(" -- ");
#endif
//set 3rd addr byte to match channel, offset 10h (Roland JV)
// subtract 1 (channels 1-16 --> 0-15)
byte channelData = MultiChan(pitch)-1;
sysxSet[7] = channelData + 0x10;
//set 4th addr byte to match note, offset 0
sysxSet[8] = (pitch - 12) % 12;
// calculate and store checksum
CheckSumSet();
//SerPrintSysx(sysxSet);
}


void SetSysxPresetAdr(byte chanPreset, byte presetType ) {
//called once for each channel/octave (chanPreset)in tuning range
//presetType: 1 = E.T., 2 = random, 3 = a row in temperCents
#ifdef DEBUG
  Serial.print("Entering SetSysxPresetAdr"); Serial.print(" Chan ");
  Serial.print(chanPreset); Serial.print(" presetType ");
  Serial.println(presetType);
  Serial. println(" -- ");
#endif
//set 3rd addr byte to match channel, offset 10h (Roland JV)
// subtract 1 (channels 1-16 --> 0-15)
byte channelData = chanPreset-1;
sysxPreset[7] = channelData + 0x10;
//set 4th addr byte to note C (= 0), offset 0
//this sysex will set 12 notes starting here:
sysxPreset[8] = 0x00;

if (presetType == 1) {
#ifdef READOUT  
  Serial.print("PresetType "); Serial.println(presetType);
#endif
Serial.println(""); Serial.print("Setting equal temperament, channel ");
Serial.println(chanPreset); Serial.println("");
  for (int k = 9 ; k < 21; k++) {
    sysxPreset[k] = initTuneVal; } 
  }
  
if (presetType == 2) {
#ifdef READOUT  
  Serial.print("PresetType "); Serial.println(presetType);
#endif
Serial.println(""); Serial.print("Randomising tuning, channel ");
Serial.println(chanPreset); Serial.println("");
  unsigned long seeD = micros();
  randomSeed(long(seeD));
  byte randLo = initTuneVal - randRange;
  byte randHi = initTuneVal + randRange + 1;
  for (int k = 9 ; k < 21; k++) {
    sysxPreset[k] = random(randLo, randHi); }
  }
  
 if (presetType == 3) {
#ifdef READOUT  
  Serial.print("PresetType "); Serial.println(presetType);
  //Serial.print("Current temperament "); Serial.println(currentTemp);
#endif
  Serial.print("Setting temperament number "); Serial.print(currentTemp);
  Serial.print(" "); Serial.println(temperamentNames[currentTemp]);
  Serial.print(" Channel "); Serial.println(chanPreset); Serial.println();
	//set zeroCentsNote to zero cents 
	int s = temperCents [currentTemp] [zeroCentsNote];
	//k is index of sysex array
	for (int k = 9 ; k < 21; k++) {
	//count columns from zero to 11 in temperCents
	int centsShifted = (temperCents [currentTemp] [k - 9]) - s;
    sysxPreset[k] = centsShifted +initTuneVal;
	} 
	
// calculate and store checksum
CheckSumPreset();
}
}

int stringToNumber(String thisString) {  // not in use
  int i, value, length;
  length = thisString.length();
  char blah[(length+1)];
  for(i=0; i<length; i++) {
    blah[i] = thisString.charAt(i);
  }
  blah[i]=0;
  value = atoi(blah);
  return value;
}

int tuneCents(byte tuneVal) {
int tCents = int(tuneVal) - 64;
return tCents;
}

void NotesOnCount() {
   Serial.print("Notes on: ");
   Serial.println(notesOn);
   if ( notesOn <= 0 )
   { digitalWrite(STATUS_LED, LOW); }
}  

void HarmonyCalc() {
#ifdef DEBUG
  Serial.println("Entering HarmonyCalc "); Serial.println();
#endif
//we come here if exactly 2 notes are on
//start with the lower of the two and set up sysex request
  byte pitchLo = min(pitchHa[0], pitchHa[1]);
  SetSysxReqAdr(pitchLo);
  
#ifdef READOUT  
  Serial.print("sending sysex pitchLo request, size ");
  Serial.print(sizeof(sysxReq)); Serial.print(" Harmony: "); Serial.println(harmonyH);
  Serial.println(" ");
#endif 
 
//send the sysex and catch the returned sysex without returning to loop()
//if the next message read isn't the requested sysex, you'll get the previous data  
  MIDI.sendSysEx(sizeof(sysxReq), sysxReq);
  while(MIDI.read() == false) {};

//HandleSystemExclusive extracts JV scale tune value harmonyVal
  byte harmonyLo = harmonyVal;
#ifdef DEBUG  
  Serial.print("harmonyLo:  "); Serial.println(harmonyLo); Serial.println();
#endif
  //get cents deviation from E.T.
  int centsLo = tuneCents(harmonyLo);

//do the same for the higher note
  byte pitchHi = max(pitchHa[0], pitchHa[1]);

  SetSysxReqAdr(pitchHi);
#ifdef DEBUG  
  Serial.print("sending sysex pitchHi request, size ");
  Serial.println(sizeof(sysxReq));
  Serial.println(" ");
#endif  

  MIDI.sendSysEx(sizeof(sysxReq), sysxReq);
  while(MIDI.read() == false) {};
  byte harmonyHi = harmonyVal;
  int centsHi = tuneCents(harmonyHi);
    
int semiTones = int(pitchHi - pitchLo);
//can't find how to assign the interval name to a variable...
//string intervalName = intervalNames[semiTones]; doesn't work

//find the cents corresponding to the interval
int centsPure = centsHarmony[semiTones];
int centsNow = centsHi - centsLo;
int centsOff = centsNow - centsPure;

Serial.println();
Serial.print(pitchLo); Serial.print(" ");
Serial.print(theNoteNames[noteIndex(pitchLo)]); Serial.print(" "); 
Serial.print(noteOctave(pitchLo)); Serial.print(" cents: "); Serial.print(centsLo);
Serial.print("  "); Serial.print(intervalNames[semiTones]);
Serial.print("  Interval if pure: "); Serial.println(centsPure);

Serial.println();
Serial.print(pitchHi); Serial.print(" ");
Serial.print(theNoteNames[noteIndex(pitchHi)]); Serial.print(" ");
Serial.print(noteOctave(pitchHi));  Serial.print(" cents: "); Serial.print(centsHi);
Serial.print("  Interval cents now: "); Serial.print(centsNow);
Serial.print("  Mistuned by: "); Serial.println(centsOff);
Serial.println();

//calculate beat frequencies
BeatFreq( pitchLo, centsLo, pitchHi, centsHi);

} 

void BeatFreq(byte pitchLo, int centsLo, byte pitchHi, int centsHi)  {

//need to work in semitones (floating hectocents!!)
const float absCentsZero = -69; //number of semitones below A440
const float powerTwo = 1.0594630943593;  //2^-12

float absCentsLo, absCentsHi;
double harmLo, harmHi;
double freqLo, freqHi;
float beats;

	absCentsLo = absCentsZero + float(pitchLo) + float(centsLo)/100;
	absCentsHi = absCentsZero + float(pitchHi) + float(centsHi)/100;
	
	freqLo = freqA * pow(powerTwo, absCentsLo);
	freqHi = freqA * pow(powerTwo, absCentsHi);
	
	Serial.print("Frequencies "); Serial.print(freqLo);
	Serial.print("  "); Serial.println(freqHi);
	Serial.println("Beats");
	
	int count = 0;
	for ( int hi = 1; hi < 7; hi++ )  {
		for ( int lo = 1; lo < 7; lo++ )  {
		
			harmLo = freqLo * lo;
			harmHi = freqHi * hi;
			beats = abs(harmHi - harmLo);
					
			if ( abs(beats)  < 20 ) {
				count = count + 1;
				Serial.print(lo); Serial.print(" : "); Serial.print(hi);
				Serial.print(" -- "); Serial.println(beats, 1);  }		
		}		
	}
if ( count == 0 ) {Serial.println("No audible beats"); }
	
}
  
  
int noteIndex(byte pitch) {
//position in scale from 0 to 11
int ntIndex = (int(pitch) - 12) % 12;
return ntIndex;
}

int noteOctave(byte pitch) {
//finds the octave of a note (Roland definition; there's no standard, but that doesn't matter)
int ntOctave = (int(pitch) / 12) - 1;
return ntOctave;
}

#ifdef INUSE 
void HandleAfterTouchPoly(byte channel, byte note, byte pressure) {
  MIDI.sendAfterTouch(pressure, midiChannel);
}
#endif
void HandleControlChange(byte channel, byte number, byte value) {
  MIDI.sendControlChange(number, value, midiChannel);
}
void HandleProgramChange(byte channel, byte number) {
  MIDI.sendProgramChange(number, midiChannel);
}
#ifdef INUSE
void HandleAfterTouchChannel(byte channel, byte pressure) {
  MIDI.sendAfterTouch(pressure, midiChannel);
}
void HandlePitchBend(byte channel, int bend) {
  MIDI.sendPitchBend (bend, midiChannel);
}
#endif
